function [pivots] = remapcc(stk_cc,varargin)
%This little function takes a cc struct from connected component analysis
%and remaps it back into image space
%Synatax:   [] = repmapcc(cc,'stks',array,'type',2)
%Input:     'stks' = stacks to be processed. in a horizontal cell array (no
%                    longer supported, if you want this roll back to
%                    version 3.5)
%           type = 1(centroid), 2 (WeightedCentroid),
%               default = 1
%Output:    pivots = the centroids or weighted centroids of the CC.

[stks,stks_ch_names,channel_paths,type,imgclass] = parse(varargin);

%first get the size of the original image
imgsize = stk_cc(1).ImageSize;

%Some people rotate the image matrix after they open it.  So, lets make
%sure
if ~isempty(stks)
    stks = reshape(stks,imgsize);
end

switch type
    case 1    %Centroid
        pivots = regionprops(stk_cc,'Centroid');  %find centroid
        %pivots_out_tmp = cat(1,pivots.Centroid);
    case 2    %WeightedCentroid
        pivots = regionprops(stk_cc,stks,'WeightedCentroid');
        %pivots_out_tmp = cat(1,pivots.WeightedCentroid);
end

centroid_stk = false(imgsize);  %create a logical array the same size as the original array
%pivots is currently a structure, convert to cell then matrix
pivots = cell2mat(struct2cell(pivots)');
%now make the pivots output by adding lum
pivots_out_tmp = pivots;
%round pivots to use as pixel indexs
pivots = round(pivots);
%now calculate the indexs for the centroid locations
[x,y,z] = size(centroid_stk);        %stack size
pivots_idx = ((pivots(:,1)-1).*x)+pivots(:,2)+((x*y).*(pivots(:,3)-1));
%modify stk_tmp
centroid_stk(pivots_idx) = 1;

pivots_out = pivots_out_tmp;
%Generate object map
object_idx = cell2mat(struct2cell(regionprops(stk_cc,'PixelIdxList'))');    %grab the indexs of all of the pixels
object_stk = false(imgsize);    %stack holder
object_stk(object_idx) = 1;     %create the image
if ~isempty(stks)
    switch imgclass
        case 'uint8'
            object_lum = stks.*uint8(object_stk);   %preserve the luminance.
        case 'uint16'
            object_lum = stks.*uint16(object_stk);   %preserve the luminance.
        case 'uint32'
            object_lum = stks.*uint32(object_stk);   %preserve the luminance.
        case 'double'
            object_lum = stks.*double(object_stk);   %preserve the luminance.
    end
   
    %Now pickup the brightness at the max
    switch imgclass
        case 'uint8'
            lum_stk = stks.*uint8(centroid_stk);
        case 'uint16'
            lum_stk = stks.*uint16(centroid_stk);
        case 'uint32'
            lum_stk = stks.*uint32(centroid_stk);
        case 'double'
            lum_stk = stks.*double(centroid_stk);
    end
end

%Now write the data out to a file and clear the memory

%dynamic path
switch type
    case 1
        dir_path = ['Centroids',datestr(now,'yyyymmdd.HHMM'),filesep];     %where to save it
    case 2
        dir_path = ['WeightedCentroids',datestr(now,'yyyymmdd.HHMM'),filesep];     %where to save it
end
mkdir(channel_paths,dir_path);      %create output directory

%initiate labeling of matrix (happy now?!)
mxlabel = {'x','y','z'};
%now save the files
warning('OFF')
mkdir([channel_paths,dir_path],stks_ch_names{1}(1:end-4));
xls_exp = dataset({pivots_out,mxlabel{:}});               %to export as a XLS need to make the data a dataset.
%luminance maintained images only work when you import an image
if ~isempty(stks)
    export(xls_exp,'File',[[channel_paths,dir_path],[stks_ch_names{1}(1:end-4),'.',num2str(size(pivots_out,1)),'.csv']],'delimiter',',')
    mkdir([channel_paths,dir_path,stks_ch_names{1}(1:end-4)],['lum_stks',filesep]);
    stk2tiff(lum_stk,stks_ch_names{1},[channel_paths,dir_path,stks_ch_names{1}(1:end-4),filesep,'lum_stks',filesep]);
end
mkdir([channel_paths,dir_path,stks_ch_names{1}(1:end-4)],['centroid_stks',filesep]);
stk2tiff(centroid_stk,stks_ch_names{1},[channel_paths,dir_path,stks_ch_names{1}(1:end-4),filesep,'centroid_stks',filesep]);
%save out the object stacks
if ~isempty(stks)
    mkdir([channel_paths,dir_path,stks_ch_names{1}(1:end-4)],['objects_lum',filesep]);
    stk2tiff(object_lum,stks_ch_names{1},[channel_paths,dir_path,stks_ch_names{1}(1:end-4),filesep,'objects_lum',filesep]);
end
mkdir([channel_paths,dir_path,stks_ch_names{1}(1:end-4)],['objects',filesep]);
stk2tiff(object_stk,stks_ch_names{1},[channel_paths,dir_path,stks_ch_names{1}(1:end-4),filesep,'objects',filesep]);

%save some info about the image
img_data = dataset(imgsize);
export(img_data,'File',[[channel_paths,filesep,dir_path],'img_data.csv'],'delimiter',',')
warning('ON')


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [stks,stks_ch_names,channel_paths,type,imgclass] = parse(input)

type = 1;
stks = [];
channel_paths = [];

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'stks'
                    stks = input{1,i+1};
                case 'type'
                    type = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end

%Get the location of the stacks you want to open
if isempty(stks)
    h = yes_no_box('title','Open Original Images','caption1','Do you want to open the original images?',...
        'caption2','Without it weighted centers is not available');
    if h    %open the original image if you want
        [stks_ch_names,channel_paths,filterindex] = uigetfile2({'*.tif','TIFF files (*.tif)';...
            '*.jpg','Jpeg files (*.jpg)';'*.gif','GIF files (*.gif)';...
            '*.bmp','Bitmap files (*.bmp)';'*.mat','Mat files (*.mat)';},'Open Stack','Multiselect','on');
        [stks] = stack_gui('filename',stks_ch_names,'pathname',channel_paths,'filterindex',filterindex);
    else
        stks_ch_names = 'remapped_ccstruct.tif';
        type = 1;  %it must be this way
    end
else
    stks_ch_names = 'remapped_ccstruct.tif';
end
if isempty(channel_paths)    %if not then we are going to need a place to save the files
    prompt_box('title','Save directory','prompt1','Where do you want to save your files?');
    channel_paths = uigetdir2('','Directory where the files are located');
end
channel_paths = [channel_paths,filesep];

%get the image class
if isempty(stks)
    imgclass = 'uint16';
else
    imgclass = class(stks);      %get image type
end

%make sure the file names are in a cell array
if ~iscell(stks_ch_names)
    stks_ch_names = {stks_ch_names};
end